#include "CodeEditor.h"

#include <QPainter>
#include <QStringListModel>
#include <QAbstractItemView>
#include <QScrollBar>

CodeEditor::CodeEditor(QWidget *parent) : QPlainTextEdit(parent), lineNumbersArea(new LineNumbersArea(this)) {
    syntaxHighlighter = new Highlighter(this->document(), this);

    QFont font;
    font.setFamily("Courier");
    font.setFixedPitch(true);
    font.setPointSize(10);
    setFont(font);

    connect(this, SIGNAL(blockCountChanged(qint32)), this, SLOT(onUpdateLNAreaWidth(qint32)));
    connect(this, SIGNAL(updateRequest(QRect, qint32)), this, SLOT(onUpdateLNArea(QRect, qint32)));
    connect(this, SIGNAL(cursorPositionChanged()), this, SLOT(highlightCurrentLine()));
    onUpdateLNAreaWidth(0);

    completer = new QCompleter(this);
    completer->setWidget(this);
    QStringList words;
    words << "module" << "input" << "output" << "endmodule";
    QStringListModel *model = new QStringListModel(words, completer);
    completer->setModel(model);
    completer->setCompletionMode(QCompleter::UnfilteredPopupCompletion);

    connect(completer, SIGNAL(activated(QString)), this, SLOT(onInsertCompletion(QString)));
}

qint32 CodeEditor::calcLNAreaWidth() {
    qint32 digits2Show = 1;
    qint32 max = qMax(1, blockCount());
    while (max >= 10){
        max /= 10;
        ++digits2Show;
    }

    return 3 + fontMetrics().horizontalAdvance(QLatin1Char('9')) * digits2Show;
}

void CodeEditor::paintLNArea(QPaintEvent *e) {
    QPainter painter(lineNumbersArea);
    painter.fillRect(e->rect(), QColor(Qt::gray).lighter(150));

    QTextBlock block = firstVisibleBlock();
    qint32 blockNumber = block.blockNumber();
    qint32 top = (qint32)blockBoundingGeometry(block).translated(contentOffset()).top();
    qint32 bot = top + (qint32)blockBoundingRect(block).height();

    while (block.isValid() && top <= e->rect().bottom()){
        if (block.isVisible() && bot >= e->rect().top()){
            QString num = QString::number(blockNumber + 1);
            painter.setPen(QColor(100, 100, 100));
            painter.setFont(font());
            painter.drawText(0, top, lineNumbersArea->width(), fontMetrics().height(), Qt::AlignRight, num);
        }

        block = block.next();
        top = bot;
        bot = top + (qint32)blockBoundingRect(block).height();
        ++blockNumber;
    }
}

void CodeEditor::resizeEvent(QResizeEvent *e) {
    QPlainTextEdit::resizeEvent(e);

    lineNumbersArea->setGeometry(QRect(contentsRect().left(), contentsRect().top(), calcLNAreaWidth(), contentsRect().height()));
}

void CodeEditor::keyPressEvent(QKeyEvent *e) {
    if ( completer->popup()->isVisible() ) {
        switch (e->key()) {
            case Qt::Key_Enter:
            case Qt::Key_Return:
            case Qt::Key_Escape:
            case Qt::Key_Tab:
                e->ignore();
                return;
        }
    }

    QPlainTextEdit::keyPressEvent(e);

    const QString completionPrefix = textUnderCursor();

    if (completionPrefix != completer->completionPrefix()) {
        completer->setCompletionPrefix(completionPrefix);
        completer->popup()->setCurrentIndex(completer->completionModel()->index(0, 0));
    }

    QRect cRect = cursorRect();
    cRect.setWidth(completer->popup()->sizeHintForColumn(0) + completer->popup()->verticalScrollBar()->sizeHint().width());
    if (!e->text().isEmpty() && completionPrefix.length() > 2)
        completer->complete(cRect);
}

QString CodeEditor::textUnderCursor() {
    QTextCursor cursor = textCursor();
    cursor.select(QTextCursor::WordUnderCursor);
    return cursor.selectedText();
}

void CodeEditor::onUpdateLNAreaWidth(qint32){
    setViewportMargins(calcLNAreaWidth(), 0, 0, 0);
}

void CodeEditor::onUpdateLNArea(const QRect &rect, qint32 dy) {
    if (dy)
        lineNumbersArea->scroll(0, dy);
    else
        lineNumbersArea->update(0, rect.y(), lineNumbersArea->width(), rect.height());

    if (rect.contains(viewport()->rect()))
        onUpdateLNAreaWidth(0);
}

void CodeEditor::highlightCurrentLine() {
    QList<QTextEdit::ExtraSelection> tmpExtraSelections;
    QTextEdit::ExtraSelection selection;
    QColor lineColor = QColor(Qt::yellow).lighter(180);
    selection.format.setBackground(lineColor);
    selection.format.setProperty(QTextFormat::FullWidthSelection, true);
    selection.cursor = textCursor();
    selection.cursor.clearSelection();
    tmpExtraSelections.prepend(selection);
    setExtraSelections(tmpExtraSelections);
}

void CodeEditor::onInsertCompletion(const QString &completion) {
    QTextCursor cursor = textCursor();
    int extra = completion.length() - completer->completionPrefix().length();
    cursor.movePosition(QTextCursor::Left);
    cursor.movePosition(QTextCursor::EndOfWord);
    cursor.insertText(completion.right(extra));
    setTextCursor(cursor);
}
