#include "Highlighter.h"

#include <QRegularExpressionMatchIterator>

Highlighter::Highlighter(QTextDocument *parent) : QSyntaxHighlighter(parent) {
    // Keywords
    const QString keywordPatterns[] = {
        QString("\\bmodule\\b"),    QString("\\bendmodule\\b"),
        QString("\\binput\\b"),     QString("\\boutput\\b"),
        QString("\\binout\\b"),     QString("\\bwire\\b"),
        QString("\\breg\\b"),       QString("\\binitial\\b"),
        QString("\\balways\\b"),    QString("\\bbegin\\b"),
        QString("\\bend\\b"),       QString("\\bnot\\b"),
        QString("\\band\\b"),       QString("\\bnand\\b"),
        QString("\\bor\\b"),        QString("\\bnor\\b"),
        QString("\\bxor\\b"),       QString("\\bxnor\\b")
    };

    QTextCharFormat keywordFormat;
    keywordFormat.setForeground(QColor(25, 49, 251));
    keywordFormat.setFontWeight(QFont::Normal);

    HighlightingRule rule;
    rule.format = keywordFormat;
    for (const QString &pattern : keywordPatterns) {
      rule.pattern = QRegularExpression(pattern);
      highlightingRules.append(rule);
    }

    // System functions
    const QString functionPatterns[] = {
        QString("\\$monitor\\b"),    QString("\\$display\\b"),
        QString("\\$strobe\\b"),     QString("\\$finish\\b"),
        QString("\\$dumpfile\\b"),   QString("\\$dumpvars\\b")
    };

    QTextCharFormat functionFormat;
    functionFormat.setForeground(QColor(43, 145, 175));
    functionFormat.setFontWeight(QFont::Normal);

    rule.format = functionFormat;
    for (const QString &pattern : functionPatterns) {
      rule.pattern = QRegularExpression(pattern);
      highlightingRules.append(rule);
    }

    // Strings
    rule.pattern = QRegularExpression("\".*\"");
    rule.format.setForeground(QColor(163, 21, 21));
    rule.format.setFontWeight(QFont::Normal);
    highlightingRules.append(rule);

    // Single-line comments
    rule.pattern = QRegularExpression("//.*$");
    rule.format.setForeground(QColor(59, 157, 59));
    rule.format.setFontWeight(QFont::Normal);
    highlightingRules.append(rule);

    // Wrong syntax
    rule.pattern = QRegularExpression("\\s*module\\s+(\\d\\w+)\\b");
    rule.format.setFontWeight(QFont::Normal);
    rule.format.setForeground(Qt::black);
    rule.format.setUnderlineColor(Qt::red);
    rule.format.setUnderlineStyle(QTextCharFormat::UnderlineStyle::WaveUnderline);
    highlightingRules.append(rule);
}

void Highlighter::highlightBlock(const QString &text) {
    for (int i = 0; i < highlightingRules.size(); ++i) {
        HighlightingRule& rule = highlightingRules[i];
        QRegularExpressionMatchIterator matchIterator = rule.pattern.globalMatch(text);
        while (matchIterator.hasNext()) {
            QRegularExpressionMatch match = matchIterator.next();
            setFormat(match.capturedStart(), match.capturedLength(), rule.format);
        }
    }
}
